package furny.swing.admin.viewer;

import java.awt.Canvas;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;

import com.jme3.app.Application;
import com.jme3.math.FastMath;
import com.jme3.system.JmeCanvasContext;

/**
 * Panel that contains the canvas, JME draws onto and some buttons.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
@SuppressWarnings("serial")
public class ViewerCanvasPanel extends JPanel {
  private final IModelViewer viewer;
  private Canvas canvas;

  private final Application app;

  private boolean initialized;

  /**
   * Instantiates a new viewer canvas panel.
   * 
   * @param app
   *          the app
   * @param viewer
   *          the viewer
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public ViewerCanvasPanel(final Application app, final IModelViewer viewer) {
    this.viewer = viewer;
    this.app = app;

    addFocusListener(new FocusAdapter() {
      @Override
      public void focusGained(final FocusEvent e) {
        checkInit();
      }
    });
  }

  /**
   * Check if initialized.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void checkInit() {
    if (!initialized) {
      app.createCanvas();

      final JmeCanvasContext context = (JmeCanvasContext) app.getContext();
      canvas = context.getCanvas();

      canvas.addComponentListener(new ComponentAdapter() {
        @Override
        public void componentResized(final ComponentEvent e) {
          updateViewerSize();
        }

        @Override
        public void componentShown(final ComponentEvent e) {
          updateViewerSize();
        }
      });

      app.startCanvas();
      viewer.setShowStatistics(false);

      setLayout(new GridBagLayout());
      final GridBagConstraints constraints = new GridBagConstraints();
      constraints.insets = new Insets(5, 5, 5, 5);
      constraints.gridx = 0;
      constraints.gridy = 0;
      constraints.weightx = 0.0d;
      constraints.weighty = 0.0d;
      constraints.fill = GridBagConstraints.NONE;
      constraints.anchor = GridBagConstraints.WEST;

      JCheckBox cb = new JCheckBox(new ActionToggleRotation());
      cb.setSelected(viewer.isRotating());

      add(cb, constraints);

      constraints.gridx++;

      cb = new JCheckBox(new ActionToggleGrid());
      cb.setSelected(viewer.isShowGrid());

      add(cb, constraints);

      constraints.gridx++;

      cb = new JCheckBox(new ActionToggleDirection());
      cb.setSelected(viewer.isShowDirection());

      add(cb, constraints);

      constraints.gridx++;

      cb = new JCheckBox(new ActionToggleScaleGrid());
      cb.setSelected(viewer.isShowScaleGrid());

      add(cb, constraints);

      constraints.gridx++;

      cb = new JCheckBox(new ActionToggleUserScaleGrid());
      cb.setSelected(viewer.isShowUserScaleGrid());

      add(cb, constraints);

      constraints.gridx++;

      cb = new JCheckBox(new ActionToggleStats());
      cb.setSelected(viewer.isShowStatistics());

      add(cb, constraints);

      constraints.gridx++;
      add(new JButton(new ActionCenter()), constraints);

      constraints.gridx++;
      add(new JButton(new ActionRotateLeft()), constraints);

      constraints.gridx++;
      add(new JButton(new ActionRotateRight()), constraints);

      constraints.gridx = 0;
      constraints.gridy++;
      constraints.weightx = 1.0d;
      constraints.weighty = 1.0d;
      constraints.gridwidth = GridBagConstraints.REMAINDER;
      constraints.fill = GridBagConstraints.BOTH;
      add(canvas, constraints);

      initialized = true;
    }
  }

  @Override
  public void doLayout() {
    super.doLayout();

    if (initialized) {
      SwingUtilities.invokeLater(new Runnable() {
        @Override
        public void run() {
          updateViewerSize();
        }
      });
    }
  }

  /**
   * Updates the viewer size.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void updateViewerSize() {
    checkInit();
    viewer.arrangeText(canvas.getWidth(), canvas.getHeight());
  }

  /**
   * Action to toggle rotation.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionToggleRotation extends AbstractAction {

    /**
     * Instantiates a new action to toggle rotation.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionToggleRotation() {
      super("Rotate");
      putValue(SHORT_DESCRIPTION, "Set rotation on/off");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      viewer.setRotating(!viewer.isRotating());
    }
  }

  /**
   * Action to toggle grid display.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionToggleGrid extends AbstractAction {

    /**
     * Instantiates a new action to toggle the grid.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionToggleGrid() {
      super("Axis");
      putValue(SHORT_DESCRIPTION, "Set axis on/off");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      viewer.setShowGrid(!viewer.isShowGrid());
    }
  }

  /**
   * Action to toggle user scale grid display.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionToggleUserScaleGrid extends AbstractAction {

    /**
     * Instantiates a new action to toggle user scale grid.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionToggleUserScaleGrid() {
      super("User scale");
      putValue(SHORT_DESCRIPTION, "Set user scale grid on/off");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      viewer.setShowUserScaleGrid(!viewer.isShowUserScaleGrid());
    }
  }

  /**
   * Action to toggle scale grid display.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionToggleScaleGrid extends AbstractAction {

    /**
     * Instantiates a new action to toggle scale grid.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionToggleScaleGrid() {
      super("Real scale");
      putValue(SHORT_DESCRIPTION, "Set calculated scale grid on/off");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      viewer.setShowScaleGrid(!viewer.isShowScaleGrid());
    }
  }

  /**
   * Action to toggle statistics display.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionToggleStats extends AbstractAction {

    /**
     * Instantiates a new action to toggle stats.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionToggleStats() {
      super("Statistics");
      putValue(SHORT_DESCRIPTION, "Set statistics on/off");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      viewer.setShowStatistics(!viewer.isShowStatistics());
    }
  }

  /**
   * Action to toggle the direction arrow display.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionToggleDirection extends AbstractAction {

    /**
     * Instantiates a new action to toggle direction arrow.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionToggleDirection() {
      super("Direction");
      putValue(SHORT_DESCRIPTION, "Set direction arrow on/off");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      viewer.setShowDirection(!viewer.isShowDirection());
    }
  }

  /**
   * Action to center the model.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionCenter extends AbstractAction {

    /**
     * Instantiates a new action to center the model.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionCenter() {
      super("Center");
      putValue(SHORT_DESCRIPTION, "Center the model");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      viewer.centerFurniture(true);
    }
  }

  /**
   * Action to rotate the model right.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionRotateRight extends AbstractAction {

    /**
     * Instantiates a new action to rotate right.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionRotateRight() {
      super("Rotate right");
      putValue(SHORT_DESCRIPTION, "Rotate the model left");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      viewer.rotateFurniture(FastMath.DEG_TO_RAD * 90, true);
    }
  }

  /**
   * Action to rotate the model left.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionRotateLeft extends AbstractAction {
    /**
     * Instantiates a new action to rotate left.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionRotateLeft() {
      super("Rotate left");
      putValue(SHORT_DESCRIPTION, "Rotate the model left");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      viewer.rotateFurniture(FastMath.DEG_TO_RAD * -90, true);
    }
  }
}
